import winston from 'winston';

const { combine, timestamp, printf, json, colorize, errors } = winston.format;

// Custom format for console logging
const consoleFormat = printf(({ level, message, timestamp, stack, ...metadata }) => {
  let msg = `${timestamp} [${level}] : ${message}`;
  if (stack) {
    msg += `\n${stack}`;
  }
  // Only stringify metadata if it's not empty
  if (Object.keys(metadata).length > 0 && Object.keys(metadata).filter(k => k !== 'level' && k !== 'message' && k !== 'timestamp').length > 0) {
    msg += `\n${JSON.stringify(metadata, null, 2)}`;
  }
  return msg;
});

const myFormat = printf(({ level, message, timestamp, ip, ...metadata }) => {
  let log = `${timestamp} [${level}]`;
  if (ip) {
    log += ` [client-ip: ${ip}]`;
  }
  log += `: ${message}`;
  if (metadata && Object.keys(metadata).length > 0) {
    log += ` \n${JSON.stringify(metadata, null, 2)}`;
  }
  return log;
});

const logger = winston.createLogger({
  level: 'info',
  format: combine(
    timestamp({ format: 'YYYY-MM-DD HH:mm:ss' }),
    errors({ stack: true }),
    myFormat
  ),
  transports: [
    new winston.transports.File({ filename: 'error.log', level: 'error' }),
    new winston.transports.File({ filename: 'combined.log' })
  ]
});

if (process.env.NODE_ENV !== 'production') {
  logger.add(new winston.transports.Console({
    format: combine(
      colorize(),
      timestamp({ format: 'YYYY-MM-DD HH:mm:ss' }),
      errors({ stack: true }),
      consoleFormat
    )
  }));
}

export default logger;
